# cli
# Copyright 2010-2011 Karl A. Knechtel.
#
# The internal logic for handling commands received by the CLI.
#
# Licensed under the Generic Non-Commercial Copyleft Software License,
# Version 1.1 (hereafter "Licence"). You may not use this file except
# in the ways outlined in the Licence, which you should have received
# along with this file.
#
# Unless required by applicable law or agreed to in writing, software 
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied. See the License for the specific language governing
# permissions and limitations under the License.


import re, inspect
from getopt import getopt, GetoptError
from exception import PEBKAC
from text import display


identifier = re.compile('^([a-zA-Z])?!([a-z][a-z-]*[a-z])?(:.*)?$')


def _nest(names):
	return ' [%s%s]' % (names[0], _nest(names[1:])) if names else ''


class _command_impl(object):
	__slots__ = [
		'__name__',
		'_func', '_options', '_long_options', '_option_mapping', '_usage',
		'_min_args', '_max_args', '_has_keywords', '_defaults'
	]


	def _add_long_option(self, opt, kind):
		# Make sure we haven't previously defined this option,
		# whether or not it has the same "requires value" semantics.
		assert opt not in self._long_options
		assert opt + kind not in self._long_options
		self._long_options.append(opt + kind)


	def _add_option(self, opt, kind):
		# This time we don't have to check the 'kind' character
		# because we're searching in a string rather than a list.
		assert opt not in self._options
		self._options += opt + kind


	def _make_usage_item(self, short_name, full_name, description):
		return ' [%s%s%s%s]' % (
			'-' + short_name if short_name else '',
			', ' if short_name and full_name else '',
			'--' + full_name if full_name else '',
			' <%s>' % description[1:] if description else ''
		)


	def __init__(self, func, params):
		self._func = func
		self._options = ''
		self._long_options = []
		self._option_mapping = {}
		usage_items = {}

		argspec = inspect.getargspec(func)
		assert argspec.args[0] == 'state'
		defaults = argspec.defaults or ()
		self._min_args = len(argspec.args) - len(defaults) - 1
		# -1 for the 'state' parameter
		assert self._min_args >= 0 # the state must not be defaulted
		self._max_args = None if argspec.varargs else len(argspec.args) - 1
		self._has_keywords = bool(argspec.keywords)
		self._defaults = list(defaults)

		# Ensure that the function can receive flags and options if provided.
		if params: assert self._has_keywords

		# Determine the options and long-options data for getopt.
		for param in params:
			short_name, full_name, description = identifier.match(param).groups()
			assert short_name or full_name

			if full_name:
				self._add_long_option(full_name, '=' if description else '')
			if short_name:
				self._option_mapping[short_name] = full_name or short_name
				self._add_option(short_name, ':' if description else '')
			usage_items[param] = self._make_usage_item(
				short_name, full_name, description
			)

		self._usage = self._func.__name__ + ''.join(
			v for k, v in sorted(usage_items.items())
		) + ''.join(
			' ' + x for x in argspec.args[1:self._min_args + 1]
		) + _nest(
			argspec.args[self._min_args + 1:] +
			([argspec.varargs + '...'] if self._max_args == None else [])
		)


	def _parse_args(self, args):
		parsed_options, parsed_args = getopt(
			args, self._options, self._long_options
		)
		parsed_args = list(parsed_args)

		min_args, max_args = self._min_args, self._max_args
		unspecified_argument = object()
		passed_args = [unspecified_argument] * min_args + self._defaults
		passed_args[:len(parsed_args)] = parsed_args
		assert unspecified_argument not in passed_args, "not enough arguments"
		assert max_args == None or len(passed_args) <= max_args, "too many arguments"

		passed_kwargs = {}
		for option, value in parsed_options:
			option = (
				option[2:] if option.startswith('--')
				else self._option_mapping[option[1:]]
			).encode('utf-8').replace('-', '_')
			passed_kwargs[option] = value

		return (passed_args, passed_kwargs)


	def cli_invoke(self, state, args):
		try:
			call_args, call_kwargs = self._parse_args(args)
		except (AssertionError, GetoptError) as e:
			raise PEBKAC(
				"Invalid arguments for command: %s.\n\nUsage: %s" % (e, self._usage)
			)
		self._func(state, *call_args, **call_kwargs)


	def __call__(self, *args, **kwargs):
		# allow transparent usage of the decorated function.
		self._func(*args, **kwargs)


	def doc(self):
		return '%s\n\n%s' % (self._usage, self._func.__doc__)


COMMANDS = {}


class command(object):
	__slots__ = ['_params']


	def __init__(self, *args):
		self._params = args


	def __call__(self, func):
		result = _command_impl(func, self._params)
		result.__name__ = func.__name__
		COMMANDS[func.__name__] = result
		return result


def execute(state, args):
	if not args: return # blank line?
	command_name, args = args[0], args[1:]

	# Can't use EAFP here because we must distinguish
	# a KeyError here from one that is raised by the user code.
	command_name = command_name.lower()
	if command_name not in COMMANDS:
		raise PEBKAC, "Unknown command."

	COMMANDS[command_name].cli_invoke(state, args)
	display('')
